<?php
/**
 * LicenZer API Handler
 */
class Licenzer_API {

    private $api_url;
    private $token;

    public function __construct($token = null) {
        $this->api_url = get_option('licenzer_api_url', LICENZER_API_URL);
        $this->token = $token ?: get_option('licenzer_reseller_token', '');
    }

    /**
     * Make API request
     */
    private function request($endpoint, $params = array()) {
        $params['token'] = $this->token;
        $url = rtrim($this->api_url, '/') . '/' . ltrim($endpoint, '/') . '?' . http_build_query($params);

        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'sslverify' => false,
        ));

        if (is_wp_error($response)) {
            return array('status' => 'error', 'message' => $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return $data ?: array('status' => 'error', 'message' => 'Invalid response');
    }

    /**
     * Register new license
     */
    public function registerLicense($ip, $month = 1, $softwareKey = null) {
        $params = array(
            'ip' => $ip,
            'month' => $month,
        );

        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }

        return $this->request('/resellerapi/register', $params);
    }

    /**
     * Activate license
     */
    public function activateLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/activate', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Deactivate license
     */
    public function deactivateLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/deactivate', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Delete license
     */
    public function deleteLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/delete', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Change license IP
     */
    public function changeIp($ip, $newIp, $softwareKey) {
        return $this->request('/resellerapi/changeiplicense', array(
            'ip' => $ip,
            'ip_new' => $newIp,
            'key' => $softwareKey,
        ));
    }

    /**
     * Get license info
     */
    public function getLicenseInfo($ip, $softwareKey) {
        return $this->request('/resellerapi/licenseinfo', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Get all licenses
     */
    public function getLicenses() {
        return $this->request('/resellerapi/getlist');
    }

    /**
     * Get discount licenses
     */
    public function getDiscountLicenses($softwareKey = null) {
        $params = array();
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        return $this->request('/resellerapi/getdiscountlicenses', $params);
    }

    /**
     * Get available softwares
     */
    public function getSoftwares() {
        return $this->request('/resellerapi/getsoftwares');
    }

    /**
     * Get reseller balance
     */
    public function getBalance() {
        return $this->request('/resellerapi/getblanace');
    }

    /**
     * Test connection
     */
    public function testConnection() {
        $result = $this->request('/resellerapi/getstatus');
        return isset($result['status']) && $result['status'] == 'success';
    }
}

