<?php
/**
 * Licenses management page
 */

if (!current_user_can('manage_options')) {
    wp_die(__('You do not have sufficient permissions to access this page.'));
}

$api = new Licenzer_API();
$licenses = $api->getLicenses();
$license_list = array();

if (isset($licenses['status']) && $licenses['status'] == 'success') {
    $license_list = $licenses['data'] ?? array();
}

// Handle actions
if (isset($_POST['action']) && wp_verify_nonce($_POST['licenzer_nonce'], 'licenzer_action')) {
    $action = sanitize_text_field($_POST['action']);
    
    if ($action == 'change_ip') {
        $license_key = sanitize_text_field($_POST['license_key']);
        $new_ip = sanitize_text_field($_POST['new_ip']);
        $software_key = sanitize_text_field($_POST['software_key']);
        $current_ip = sanitize_text_field($_POST['current_ip']);
        
        if (filter_var($new_ip, FILTER_VALIDATE_IP)) {
            $result = $api->changeIp($current_ip, $new_ip, $software_key);
            if (isset($result['status']) && $result['status'] == 'success') {
                echo '<div class="notice notice-success"><p>' . __('IP changed successfully', 'licenzer') . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html($result['message'] ?? __('Failed to change IP', 'licenzer')) . '</p></div>';
            }
        } else {
            echo '<div class="notice notice-error"><p>' . __('Invalid IP address', 'licenzer') . '</p></div>';
        }
    }
}

// Refresh licenses after action
$licenses = $api->getLicenses();
if (isset($licenses['status']) && $licenses['status'] == 'success') {
    $license_list = $licenses['data'] ?? array();
}

?>
<div class="wrap licenzer-licenses">
    <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

    <table class="wp-list-table widefat fixed striped">
        <thead>
            <tr>
                <th><?php _e('License Key', 'licenzer'); ?></th>
                <th><?php _e('IP Address', 'licenzer'); ?></th>
                <th><?php _e('Software', 'licenzer'); ?></th>
                <th><?php _e('Status', 'licenzer'); ?></th>
                <th><?php _e('Expiry Date', 'licenzer'); ?></th>
                <th><?php _e('Actions', 'licenzer'); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($license_list)): ?>
                <tr>
                    <td colspan="6" class="text-center"><?php _e('No licenses found', 'licenzer'); ?></td>
                </tr>
            <?php else: ?>
                <?php foreach ($license_list as $license): ?>
                <tr>
                    <td><code><?php echo esc_html($license['license_key'] ?? 'N/A'); ?></code></td>
                    <td><?php echo esc_html($license['ip'] ?? 'N/A'); ?></td>
                    <td><?php echo esc_html($license['software_name'] ?? $license['key'] ?? 'N/A'); ?></td>
                    <td>
                        <span class="status-badge status-<?php echo ($license['status'] ?? 0) ? 'active' : 'inactive'; ?>">
                            <?php echo ($license['status'] ?? 0) ? __('Active', 'licenzer') : __('Inactive', 'licenzer'); ?>
                        </span>
                    </td>
                    <td><?php echo esc_html($license['end_at'] ?? $license['Expired date'] ?? 'N/A'); ?></td>
                    <td>
                        <button class="button button-small change-ip-btn" 
                                data-license-key="<?php echo esc_attr($license['license_key'] ?? ''); ?>"
                                data-current-ip="<?php echo esc_attr($license['ip'] ?? ''); ?>"
                                data-software-key="<?php echo esc_attr($license['key'] ?? ''); ?>">
                            <?php _e('Change IP', 'licenzer'); ?>
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Change IP Modal -->
<div id="change-ip-modal" style="display: none;">
    <div class="licenzer-modal-content">
        <h2><?php _e('Change IP Address', 'licenzer'); ?></h2>
        <form method="post" id="change-ip-form">
            <?php wp_nonce_field('licenzer_action', 'licenzer_nonce'); ?>
            <input type="hidden" name="action" value="change_ip">
            <input type="hidden" name="license_key" id="modal-license-key">
            <input type="hidden" name="software_key" id="modal-software-key">
            <input type="hidden" name="current_ip" id="modal-current-ip">
            
            <table class="form-table">
                <tr>
                    <th><label><?php _e('Current IP', 'licenzer'); ?></label></th>
                    <td><input type="text" id="modal-current-ip-display" class="regular-text" readonly></td>
                </tr>
                <tr>
                    <th><label for="new-ip"><?php _e('New IP Address', 'licenzer'); ?></label></th>
                    <td>
                        <input type="text" name="new_ip" id="new-ip" class="regular-text" required pattern="^([0-9]{1,3}\.){3}[0-9]{1,3}$" placeholder="192.168.1.1">
                        <p class="description"><?php _e('Enter the new IP address for this license', 'licenzer'); ?></p>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <button type="submit" class="button button-primary"><?php _e('Change IP', 'licenzer'); ?></button>
                <button type="button" class="button cancel-modal"><?php _e('Cancel', 'licenzer'); ?></button>
            </p>
        </form>
    </div>
</div>

