<?php

namespace Licenzer\Licenzer;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class LicenzerApi
{
    private $apiUrl;
    private $token;
    private $client;

    public function __construct($apiUrl, $token)
    {
        $this->apiUrl = rtrim($apiUrl, '/');
        $this->token = $token;
        $this->client = new Client([
            'timeout' => 30,
            'verify' => false,
        ]);
    }

    /**
     * Make API request
     */
    private function request(string $endpoint, array $params = []): array
    {
        $params['token'] = $this->token;
        $url = $this->apiUrl . '/' . ltrim($endpoint, '/') . '?' . http_build_query($params);

        try {
            $response = $this->client->get($url);
            $data = json_decode($response->getBody()->getContents(), true);
            return $data ?: ['status' => 'error', 'message' => 'Invalid response'];
        } catch (GuzzleException $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Register new license
     */
    public function registerLicense(string $ip, int $month = 1, ?string $softwareKey = null): array
    {
        $params = [
            'ip' => $ip,
            'month' => $month,
        ];

        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }

        return $this->request('/resellerapi/register', $params);
    }

    /**
     * Activate license
     */
    public function activateLicense(string $ip, string $softwareKey): array
    {
        return $this->request('/resellerapi/activate', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }

    /**
     * Deactivate license
     */
    public function deactivateLicense(string $ip, string $softwareKey): array
    {
        return $this->request('/resellerapi/deactivate', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }

    /**
     * Delete license
     */
    public function deleteLicense(string $ip, string $softwareKey): array
    {
        return $this->request('/resellerapi/delete', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }

    /**
     * Change license IP
     */
    public function changeIp(string $ip, string $newIp, string $softwareKey): array
    {
        return $this->request('/resellerapi/changeiplicense', [
            'ip' => $ip,
            'ip_new' => $newIp,
            'key' => $softwareKey,
        ]);
    }

    /**
     * Get license info
     */
    public function getLicenseInfo(string $ip, string $softwareKey): array
    {
        return $this->request('/resellerapi/licenseinfo', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }

    /**
     * Get all licenses
     */
    public function getLicenses(): array
    {
        return $this->request('/resellerapi/getlist');
    }

    /**
     * Get discount licenses
     */
    public function getDiscountLicenses(?string $softwareKey = null): array
    {
        $params = [];
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        return $this->request('/resellerapi/getdiscountlicenses', $params);
    }

    /**
     * Get available softwares
     */
    public function getSoftwares(): array
    {
        return $this->request('/resellerapi/getsoftwares');
    }

    /**
     * Get reseller balance
     */
    public function getBalance(): array
    {
        return $this->request('/resellerapi/getblanace');
    }

    /**
     * Test connection
     */
    public function testConnection(): bool
    {
        $result = $this->request('/resellerapi/getstatus');
        return isset($result['status']) && $result['status'] == 'success';
    }
}

